/*
  Copyright (c) 2003, Purdue University
  All rights reserved.

  Redistribution and use in source and binary forms are permitted provided
  that:

  (1) source distributions retain this entire copyright notice and comment,
      and
  (2) distributions including binaries display the following acknowledgement:

         "This product includes software developed by Purdue University."

      in the documentation or other materials provided with the distribution
      and in all advertising materials mentioning features or use of this
      software.

  The name of the University may not be used to endorse or promote products
  derived from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.

  This software was developed by:
     Curtis Smith

     Purdue University
     Engineering Computer Network
     1285 Electrical Engineering Building
     West Lafayette, Indiana 47907-1285 U.S.A.

  Send all comments, suggestions, or bug reports to:
     software@ecn.purdue.edu
*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <ctype.h>
#include <openssl/des.h>
#include <sys/param.h>

/*
	ANSI X9.71 password generator

	Let E[k](X) be a triple-DES encryption of X with key k.
	V(0) is a secret 64 bit seed.
	T is a timestampt.

	To generate the random key R[i], calculate:

	R[i] = E[k]( E[k](T) ^ V[i] )

	To calculate V[i+1]:

	V[i+1] = E[k]( E[k](T) ^ R[i] )
*/	

static char ch_salt[] =
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789./";
static char ch_alpha[] =
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
static char ch_symbol[] =
"!\"#$%&'()*+,-./:;<=>?@[]^_`{|}~";
static char ch_number[] =
"0123456789";

#define PICK(v,a)	(a)[(v/2) % (sizeof(a)-1)]

#define FALSE		0
#define TRUE		1

int main(int argc, char ** argv)
{
	char * home;
	char ch;
	char password[8+1];
	char seed[8];
	char seedfile[MAXPATHLEN+1];
	des_cblock encrypted_tod;
	des_cblock key[2];
	des_cblock result;
	des_cblock tod;
	des_key_schedule key_schedule[2];
	int count;
	int fd;
	int i;
	int number;
	int pass_count;
	int pass_found = FALSE;
	int pass_hosed = FALSE;
	int pass_read = FALSE;
	int symbol;
	time_t now;

	/* Get password count */
	if (argc > 3) {
		pass_count = atoi(argv[3]);
		if (pass_count == 0)
			pass_count = 1;
	} else
		pass_count = 1;

	/* Generate name of seed file */
	home = getenv("HOME");
	if (home == NULL)
		home = "";
	snprintf(seedfile, sizeof(seedfile), "%s/.seed", home);

	/* Read in the last seed */
	fd = open(seedfile, O_RDWR);
	if (fd < 0) {
		if (errno != ENOENT) {
			perror(seedfile);
			return 1;
		}
		fd = open(seedfile, O_WRONLY|O_CREAT, 0600);
		if (fd < 0) {
			perror(seedfile);
			return 1;
		}
		fprintf(stderr, "%s: Creating new seed file \"%s\"\n", argv[0], seedfile);
	} else {
		count = read(fd, seed, sizeof(seed));
		if (count != sizeof(seed)) {
			if (count < 0) {
				perror(seedfile);
				return 1;
			}
			fprintf(stderr, "%s: short random number seed file \"%s\"\n", argv[0], seedfile);
		}
	}

	/* Get the time of day */
	time(&now);

	/* Loop until complete */
	while (pass_hosed == FALSE) {

		/* Use the time of day in both halfs of key */
		memcpy(tod, &now, sizeof(now));
		memcpy(tod+sizeof(now), &now, sizeof(now));

		/* Prompt for password */
		if (pass_read == FALSE) {
			pass_read = TRUE;
			if (argc > 1) {
				des_string_to_key(argv[1], &key[0]);
			} else {
				if (des_read_password(&key[0], "Password 1: ", 0)) {
					fprintf(stderr, "%s: Could not read password\n", argv[0]);
					return 1;
				}
			}
			if (argc > 2) {
				des_string_to_key(argv[2], &key[1]);
			} else {
				if (des_read_password(&key[1], "Password 2: ", 0)) {
					fprintf(stderr, "%s: Could not read password\n", argv[0]);
					return 1;
				}
			}
		}

		/* Set key schedule */
		des_set_key(&key[0], key_schedule[0]);
		des_set_key(&key[1], key_schedule[1]);

		/* Compute E(time of day) using prompted keys */
		des_ecb2_encrypt(&tod, &encrypted_tod, key_schedule[0], key_schedule[1], 1);

		/* Compute E(time of day) XOR seed */
		for (i = 0; i < sizeof(seed); i++)
			encrypted_tod[i] ^= seed[i];

		/* Compute E(E(time of day) XOR seed) */
		des_ecb2_encrypt(&encrypted_tod, &result, key_schedule[0], key_schedule[1], 1);

		/* Compute E(time of day) XOR result */
		for (i = 0; i < sizeof(result); i++)
			encrypted_tod[i] ^= result[i];

		/* Compute E(E(time of day) XOR result) */
		des_ecb2_encrypt(&encrypted_tod, &result, key_schedule[0], key_schedule[1], 1);

		/* Skip if password already found */
		if (pass_found < pass_count) {

			/* Output first two characters from salt selection */
			password[0] = PICK(result[0], ch_salt);
			password[1] = PICK(result[1], ch_salt);

			/* Pick places for number and symbol */
			symbol = (result[2]/2) % 6;
			number = (result[2]/2) % 5;
			if (symbol == number)
				number = 5;
			for (i = 2; i < sizeof(result); i++) {
				if (i-2 == symbol)
					ch = PICK(result[i], ch_symbol);
				else if (i-2 == number)
					ch = PICK(result[i], ch_number);
				else
					ch = PICK(result[i], ch_alpha);
				password[i] = ch;
			}
			password[8] = '\0';

			/* Check password */
			if (FascistCheck(password, CRACKLIB_DICTPATH) == NULL) {
				pass_found++;
				puts(password);
			}
		} else
			pass_hosed = TRUE;

		/* Result is the new seed */
		memcpy(seed, result, sizeof(seed));
	}

	/* Store new seed */
	if (lseek(fd, 0, SEEK_SET) != 0) {
		perror(seedfile);
		return 1;
	}
	if (write(fd, seed, sizeof(seed)) != sizeof(seed)) {
		perror(seedfile);
		return 1;
	}
	close(fd);

	/* Success */
	return 0;
}
